import argparse
import configparser
import time
from datetime import datetime
from rdaf import rdafutils
import json
import logging
import os
import socket
import string
from typing import Callable, Any, List

import yaml
import rdaf
from rdaf.component import Component, OtherCategoryOrder, dockerregistry, \
    _comma_delimited_to_list, _list_to_comma_delimited, run_potential_ssh_command, \
                            do_potential_scp, run_command, execute_command
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.rdafutils import cli_err_exit
import rdaf.component.haproxy as haproxy_comp

COMPONENT_NAME = 'rda_worker'
logger = logging.getLogger(__name__)


class Worker(Component):
    _option_host = 'host'
    _option_site_group = 'site'


    def __init__(self):
        super().__init__(COMPONENT_NAME, 'rda_worker', category='other',
                         category_order=OtherCategoryOrder.WORKER.value)

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name in [self._option_host, self._option_site_group]:
            return _comma_delimited_to_list
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name in [self._option_host, self._option_site_group]:
            return _list_to_comma_delimited
        return None

    def get_conf_dir(self) -> os.path:
        return os.path.join(self.get_rdaf_install_root(), 'config')

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_host] = None
        default_configs[self._option_site_group] = []
        return default_configs

    def gather_minimal_setup_inputs(self, cmd_args, config_parser):
        worker_configs = self._init_default_configs()
        worker_configs[self._option_host] = [self.get_default_host()]
        self._mark_configured(worker_configs, config_parser)

    def gather_setup_inputs(self, cmd_args: argparse.Namespace,
                            config_parser: configparser.ConfigParser):
        worker_configs = self._init_default_configs()
        default_host_name = socket.gethostname()
        err_msg = 'No Worker host specified. Use --worker-host ' \
                  + 'to specify one'
        desc = 'What are the host(s) on which you want the Worker to be installed?'
        worker_hosts = self._parse_or_prompt_hosts(cmd_args.worker_host,
                                                   default_host_name,
                                                   err_msg, desc,
                                                   'Worker host(s)',
                                                   cmd_args.no_prompt)
        worker_configs[self._option_host] = worker_hosts

        # Store site groups for the hosts configured during setup
        # Check if site_group is provided in cmd_args, otherwise use default
        site_group = getattr(cmd_args, 'site_group', None) or 'rda-site-01'
        site_mappings = [f"{host}/{site_group}" for host in worker_hosts]
        worker_configs[self._option_site_group] = site_mappings
        logger.info(f"Configured site group '{site_group}' for worker hosts: {worker_hosts}")
        self._mark_configured(worker_configs, config_parser)

    def do_setup(self, cmd_args, config_parser):
        for host in self.get_hosts():
            logger.debug("copying logging yaml worker on host {}".format(host))
            self.copy_logging_config(config_parser, host, 'rda_worker')

    def add_worker_host(self, cmd_args: argparse.Namespace, host: str, config_parser: configparser.ConfigParser):
        site_group = getattr(cmd_args, 'site_group', None)
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME]
        )
        if host not in all_known_hosts:
            self.setup_install_root_dir_hierarchy(host, config_parser)
        self._copy_configs(host)
        self.configs[self._option_host].append(host)

        if not site_group:
            site_group = 'rda-site-01' 

        logger.info(f"Setting site group '{site_group}' for host {host}")
        self.set_site_group_for_host(host, site_group, config_parser)

        # doing a docker login
        docker_registry = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
        docker_registry.docker_login(host, config_parser)

    def _copy_configs(self, host):
        network_config = os.path.join(self.get_conf_dir(), 'network_config', 'config.json')
        if not Component.is_local_host(host):
            logger.info('Creating directory ' + os.path.dirname(network_config))
            do_potential_scp(host, network_config, network_config)

    def get_k8s_component_name(self):
        return 'rda-worker'

    def get_hosts(self) -> list:
        return self.configs[self._option_host]

    def get_site_group_for_host(self, host: str, config_parser: configparser.ConfigParser = None) -> str:
        """Get the site group for a specific host from the configuration"""
        if config_parser:
            # Reload configs from file to get latest configuration
            self.load_config(config_parser)

        site_mappings = self.configs[self._option_site_group]
        for mapping in site_mappings:
            if '/' in mapping:
                mapped_host, site_group = mapping.split('/', 1)
                if mapped_host.strip() == host:
                    return site_group.strip()
        # Return default site group if not found
        logger.debug(f"No specific site group configured for host {host}, using default: rda-site-01")
        return 'rda-site-01'

    def set_site_group_for_host(self, host: str, site_group: str, config_parser: configparser.ConfigParser):
        """Set or update the site group for a specific host in the configuration"""
        site_mappings = self.configs[self._option_site_group]
        updated = False

        # Update existing mapping if found
        for i, mapping in enumerate(site_mappings):
            if '/' in mapping:
                mapped_host, _ = mapping.split('/', 1)
                if mapped_host.strip() == host:
                    site_mappings[i] = f"{host}/{site_group}"
                    updated = True
                    break

        # Add new mapping if not found
        if not updated:
            site_mappings.append(f"{host}/{site_group}")

        self.configs[self._option_site_group] = site_mappings
        self.store_config(config_parser)
        logger.info(f"Updated site group for host {host} to {site_group}")

    def get_deployment_file_path(self, host: str) -> os.path:
        return os.path.join('/opt', 'rdaf', 'deployment-scripts', host, 'worker.yaml')

    def get_deployment_replacements(self, cmd_args: argparse.Namespace, host: str, config_parser: configparser.ConfigParser = None) -> dict:
        replacements = self._get_docker_repo()
        replacements['TAG'] = cmd_args.tag
        replacements['WORKER_GROUP'] = self.get_site_group_for_host(host, config_parser)
        haproxy_advertised_host = COMPONENT_REGISTRY.require(haproxy_comp.COMPONENT_NAME).get_internal_access_host()
        replacements['ADVERTISED_HOST'] = haproxy_advertised_host
        return replacements

    def create_compose_file_worker(self, cmd_args: argparse.Namespace, host: str, config_parser: configparser.ConfigParser = None, update_hosts=True):
        compose_file = self.get_deployment_file_path(host)
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(),
                                 'worker.yaml')

        replacements = self.get_deployment_replacements(cmd_args, host, config_parser)
        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        original_values = yaml.safe_load(original_content)
        self.apply_user_inputs('rda_worker', original_values['services']['rda_worker'])

        ipv6_network_block = self.generate_ipv6_network_block()
        if ipv6_network_block:
            if 'networks' not in original_values['services']['rda_worker']:
                original_values['services']['rda_worker']['networks'] = ["rda_worker"]
            if 'networks' not in original_values:
                original_values['networks'] = {'rda_worker': ipv6_network_block}

        # Create directory if it doesn't exist
        run_command('mkdir -p ' + os.path.dirname(compose_file))
        with open(compose_file, 'w') as f:
            yaml.safe_dump(original_values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        # Copy the compose file only to the specific host it was created for
        if update_hosts and not Component.is_local_host(host):
            do_potential_scp(host, compose_file, compose_file)

    def get_images_involved(self):
        return ['ubuntu-rda-worker-all']

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
            # Create compose file with host-specific site group and copy to that host
            self.create_compose_file_worker(cmd_args, host, config_parser, update_hosts=True)
            deployment_file = self.get_deployment_file_path(host)
            command = '/usr/local/bin/docker-compose --project-name deployment-scripts -f {file} up -d '.format(file=deployment_file)
            self.copy_logging_config(config_parser, host, 'rda_worker')
            site_group = self.get_site_group_for_host(host, config_parser)  # For logging purposes
            logger.info("Installing worker on host {} with site group {}".format(host, site_group))
            run_potential_ssh_command(host, command, config_parser)

    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if cmd_args.rolling:
            self.handle_rolling_upgrade(cmd_args, config_parser)
            return

        for host in self.get_hosts():
            # Create compose file with host-specific site group and copy to that host
            self.create_compose_file_worker(cmd_args, host, config_parser, update_hosts=True)
            deployment_file = self.get_deployment_file_path(host)
            command = '/usr/local/bin/docker-compose --project-name deployment-scripts -f {file} up -d '.format(file=deployment_file)
            site_group = self.get_site_group_for_host(host, config_parser)  # For logging purposes
            logger.info("Upgrading worker on host {} with site group {}".format(host, site_group))
            self.copy_logging_config(config_parser, host, 'rda_worker')
            run_potential_ssh_command(host, command, config_parser)


    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
            deployment_file = self.get_deployment_file_path(host)
            if not os.path.exists(deployment_file):
                continue
            command = '/usr/local/bin/docker-compose --project-name deployment-scripts -f ' + deployment_file + ' up -d '
            logger.info("Creating worker on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in reversed(self.get_hosts()):
            deployment_file = self.get_deployment_file_path(host)
            if not os.path.exists(deployment_file):
                continue
            logger.info("Deleting worker on host {}".format(host))
            command = '/usr/local/bin/docker-compose --project-name deployment-scripts -f ' + deployment_file + ' rm -fs rda_worker'
            run_potential_ssh_command(host, command, config_parser)

    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        command = f'kubectl get deployment -n {namespace} -l app_component=rda-worker -o json'
        ret, stdout, stderr = execute_command(command)
        components = json.loads(stdout)
        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if os.path.exists(values_file):
            with open(values_file) as f:
                data = yaml.safe_load(f)
        for component in components["items"]:
            comp = component["metadata"]["labels"]["app_component"]
            replicas = data['rda_worker']['replicas']
            logger.info(f"Creating service: {comp} ")
            command = f'kubectl scale deployment.apps/{comp} -n {namespace} --replicas={replicas}'
            run_command(command)
            return

    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        command = f'kubectl get deployment -n {namespace} -l app_component=rda-worker -o json'
        ret, stdout, stderr = execute_command(command)
        components = json.loads(stdout)
        for component in components["items"]:
            comp = component["metadata"]["labels"]["app_component"]
            logger.info(f"Deleting service: {comp} ")
            command = f'kubectl scale deployment.apps/{comp} -n {namespace} --replicas=0'
            run_command(command)
            if cmd_args.force:
                    delete_pods_command = f'kubectl delete pods -n {namespace} -l app_component={comp} --grace-period=0 --force'
                    run_command(delete_pods_command)
            return

    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
            deployment_file = self.get_deployment_file_path(host)
            if not os.path.exists(deployment_file):
                continue
            command = '/usr/local/bin/docker-compose --project-name deployment-scripts -f ' + deployment_file + ' start '
            logger.info("Starting worker on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in reversed(self.get_hosts()):
            deployment_file = self.get_deployment_file_path(host)
            if not os.path.exists(deployment_file):
                continue
            logger.info("stopping worker on host {}".format(host))
            command = '/usr/local/bin/docker-compose --project-name deployment-scripts -f ' + deployment_file + ' stop rda_worker'
            run_potential_ssh_command(host, command, config_parser)

    def get_k8s_install_args(self, cmd_args):
        args = '--set tag={} '.format(cmd_args.tag)

        # adding the timestamp for daily tags which will work for upgrade scenarios to pull the
        # latest image, with a change to the chart templates.
        if cmd_args.tag == 'daily':
            timestamp = datetime.now().strftime("%Y-%m-%d-%H:%M:%S")
            args += ' --set {service}.extraEnvs[0].name=timestamp,{service}.extraEnvs[0].value={value}' \
                .format(service='rda_worker', value=str(timestamp))
        return args

    def k8s_pull_images(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != 'k8s':
            return
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        for host in self.get_hosts():
            logger.info(f'Pulling {self.component_name} images on host {host}')
            docker_pull_command = f'docker pull {docker_repo}/ubuntu-rda-worker-all:{cmd_args.tag}'
            run_potential_ssh_command(host, docker_pull_command, config_parser)

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        install_command = 'helm install --create-namespace -n {} -f {} {} {} {} '\
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(),
                    deployment_path)
        run_command(install_command)

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        upgrade_command = 'helm upgrade --install --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(),
                    deployment_path)
        run_command(upgrade_command)

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        statuses = []
        for host in self.get_hosts():
            deployment_file = self.get_deployment_file_path(host)
            if not os.path.exists(deployment_file):
                logger.debug("No platform services deployed.")
                return statuses
            component_status = dict()
            statuses.append(component_status)
            component_status['component_name'] = self.get_name()
            component_status['host'] = host
            try:
                with Component.new_docker_client(host, config_parser) as docker_client:
                    containers = self.find_component_container_on_host(docker_client, all_states=True)
                    if len(containers) == 0:
                        logger.debug(
                            'No container found for ' + self.get_name() + ' on host ' + host)
                        component_status['containers'] = []
                    else:
                        component_status['containers'] = containers
            except Exception:
                logger.debug('Failed to get status of ' + self.get_name() + ' on host ' + host,
                             exc_info=1)
                # set the status as error
                component_status['status'] = {'error': True, 'message': 'Unknown'}

        return statuses

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        namespace = self.get_namespace(config_parser)
        statuses = []
        status_command = 'kubectl get pods -n {} -l app_component={} -o json'\
            .format(namespace, self.get_k8s_component_name())
        ret, stdout, stderr = execute_command(status_command)
        if ret != 0:
            cli_err_exit("Failed to get status of event-gateway, due to: {}.".format(str(stderr)))
        result = json.loads(str(stdout))
        items = result['items']
        hosts = self.get_hosts()
        for host in hosts:
            deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
            if not items and os.path.exists(deployment_path):
                    statuses.append({
                        'component_name': self.get_k8s_component_name(),
                        'host': host,
                        'containers': [{
                            'Id': 'N/A',
                            'Image': 'N/A',
                            'State': 'N/A',
                            'Status': 'Not Provisioned'
                        }]
                    })
        else:
            for item in items:
                pod = dict()
                statuses.append(pod)
                pod['component_name'] = self.get_k8s_component_name()
                pod['host'] = item['status'].get('hostIP', 'Unknown')
                pod['containers'] = []
                if 'containerStatuses' in item['status']:
                    for entry in item['status']['containerStatuses']:
                        if 'containerID' in entry.keys():
                            container = dict()
                            container['Id'] = entry['containerID']
                            container['Image'] = entry['image']
                            for key in entry['state'].keys():
                                container['State'] = key
                                if key == 'running':
                                    container['Status'] = self.get_container_age(entry['state'][key]['startedAt'])
                                else:
                                    container['Status'] = key
                                break
                            pod['containers'].append(container)
            return statuses

    def handle_rolling_upgrade(self, cmd_args, config_parser):
        if len(self.get_hosts()) == 1:
            logger.info("Worker is deployed on single host.. continuing with normal upgrade.")
            cmd_args.rolling = False
            self.upgrade(cmd_args, config_parser)
            return
        logger.info("Collecting worker details for rolling upgrade ")
        # get all worker containers
        completed_hosts = []
        host_containers = {}
        container_id_host = {}
        for host in self.get_hosts():
            try:
                with Component.new_docker_client(host, config_parser) as docker_client:
                    containers = self.find_component_container_on_host(docker_client)
                    if len(containers) == 0:
                        logger.debug(
                            'No worker container found for ' + self.get_name() + ' on host ' + host)
                    else:
                        host_containers[host] = containers[0]
                        container_id_host[containers[0]['Id'][:12]] = host
            except Exception:
                logger.warning('Failed to get status of ' + self.get_name() + ' on host ' + host,
                               exc_info=1)

        if len(host_containers) < 2:
            if not rdafutils.query_yes_no(f"Warning: Only {len(host_containers)} running worker(s) found. "
                                         f"Minimum of two running workers recommended for rolling upgrade. Continue anyway?"):
                cli_err_exit("Exiting rolling upgrade, minimum of two running workers are needed.")

        # checking the rdac pods availability
        return_code, stdout, stderr = execute_command('rdac pods --type worker --json --show_maintenance')
        sanitized_output = rdafutils.cleanup_stdout(stdout)
        rdac_pods = json.loads(sanitized_output)
        if len(rdac_pods) < 2:
            if not rdafutils.query_yes_no(f"Warning: Only {len(rdac_pods)} worker pod(s) available. "
                                         f"Minimum of two worker pods recommended for rolling upgrade. Continue anyway?"):
                cli_err_exit("Exiting rolling upgrade, minimum of two workers pods needs to be available.")

        container_to_pod = {}
        for pod in rdac_pods:
            container_to_pod[pod.get("hostname")] = pod
            if pod.get('is_maintenance'):
                logger.info("Following worker container is in maintenance mode")
                table = [["Pod ID", "Pod Type", "Version", "Age", "Hostname", "Maintenance", "Pod Status"]]
                age = datetime.fromisoformat(pod.get("now")) - datetime.fromisoformat(pod.get("started_at"))
                age = str(age).rsplit(".", 1)[0]
                table.append([pod.get("pod_id", ""), pod.get("pod_type", ""), pod.get("build_tag", ""),
                              str(age), pod.get("hostname", ""), pod.get("is_maintenance"),
                              pod.get("pod_ready")])
                rdafutils.print_tabular(table[0], table[1:])
                host = container_id_host[pod.get("hostname")]
                self.create_compose_file_worker(cmd_args, host, config_parser, update_hosts=False)
                deployment_file = self.get_deployment_file_path(host)
                if not Component.is_local_host(host):
                    do_potential_scp(host, deployment_file, deployment_file)
                command = '/usr/local/bin/docker-compose --project-name deployment-scripts -f {file} up -d '.format(file=deployment_file)

                logger.info("Upgrading worker on host {}".format(host))
                self.copy_logging_config(config_parser, host, 'rda_worker')
                run_potential_ssh_command(host, command, config_parser)
                self.wait_for_ru_container_to_join_pods(host, config_parser)
                completed_hosts.append(host)

        for host in self.get_hosts():
            if host in completed_hosts:
                continue
            container = host_containers[host]
            container_id = container['Id'][:12]

            pod = {}
            for entry in rdac_pods:
                # here hostname will be representing pod's unique short uuid
                if entry.get("hostname") == container_id:
                    pod = entry
                    break

            if not pod:
                logger.warning(f"WARNING!!! worker container {container_id} on {host} has not found in rdac pods")
                return

            logger.info(f"Rolling upgrade worker on {host}")
            table = [["Pod ID", "Pod Type", "Version", "Age", "Hostname", "Maintenance", "Pod Status"]]
            age = datetime.fromisoformat(pod.get("now")) - datetime.fromisoformat(pod.get("started_at"))
            age = str(age).rsplit(".", 1)[0]
            table.append([pod.get("pod_id", ""), pod.get("pod_type", ""), pod.get("build_tag", ""), str(age),
                          pod.get("hostname", ""), pod.get("is_maintenance"), pod.get("pod_ready")])

            rdafutils.print_tabular(table[0], table[1:])

            if not rdafutils.query_yes_no("Continue moving above pod to maintenance mode?"):
                return
            pod_id = pod.get("pod_id")
            logger.info(f'Initiating maintenance mode for pod {pod_id}')
            execute_command(f"rdac maintenance start --ids {pod_id}")
            # checking if pod joined the maintenance mode
            maint_pod = {}
            retry = 0
            while retry < 10:
                retry += 1
                return_code, stdout, stderr = execute_command('rdac pods --show_maintenance --json --type worker')
                sanitized_output = rdafutils.cleanup_stdout(stdout)
                rdac_pods = json.loads(sanitized_output)
                for pod in rdac_pods:
                    if pod.get("hostname") == container_id and pod.get('is_maintenance'):
                        maint_pod = pod
                        break
                if maint_pod:
                    break
                logger.info(f"Waiting for worker to be moved to maintenance.")
                time.sleep(10)
            if not maint_pod:
                if not rdafutils.query_yes_no(f"Warning: Failed to move worker on host {host} to maintenance mode. Continue anyway?"):
                    cli_err_exit(f'Failed to move worker on host {host} to maintenance.')

            logger.info("Following worker container is in maintenance mode")
            table = [["Pod ID", "Pod Type", "Version", "Age", "Hostname", "Maintenance", "Pod Status"]]
            age = datetime.fromisoformat(maint_pod.get("now")) - datetime.fromisoformat(maint_pod.get("started_at"))
            age = str(age).rsplit(".", 1)[0]
            table.append([maint_pod.get("pod_id", ""), maint_pod.get("pod_type", ""), maint_pod.get("build_tag", ""),
                          str(age), maint_pod.get("hostname", ""), maint_pod.get("is_maintenance"),
                          maint_pod.get("pod_ready")])
            rdafutils.print_tabular(table[0], table[1:])
            timeout = cmd_args.timeout
            logger.info(f"Waiting for timeout of {timeout} seconds.")
            time.sleep(timeout)
            self.create_compose_file_worker(cmd_args, host, config_parser, update_hosts=False)
            deployment_file = self.get_deployment_file_path(host)
            if not Component.is_local_host(host):
                do_potential_scp(host, deployment_file, deployment_file)
            command = '/usr/local/bin/docker-compose --project-name deployment-scripts -f {file} up -d '.format(file=deployment_file)

            logger.info("Upgrading worker on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)
            completed_hosts.append(host)
            self.wait_for_ru_container_to_join_pods(host, config_parser)

        logger.info("Completed worker rolling upgrade.")

    def wait_for_ru_container_to_join_pods(self, host, config_parser):
        logger.info("Waiting for upgraded worker to join rdac pods.")
        try:
            with Component.new_docker_client(host, config_parser) as docker_client:
                containers = self.find_component_container_on_host(docker_client, all_states=True)
                if len(containers) == 0:
                    cli_err_exit(
                        'No worker container found for ' + self.get_name() + ' on host ' + host)
                else:
                    container = containers[0]
        except Exception:
            logger.debug('Failed to get status of ' + self.get_name() + ' on host ' + host,
                         exc_info=1)

        container_id = container['Id'][:12]
        retry = 0
        while retry < 6:
            retry += 1
            return_code, stdout, stderr = execute_command('rdac pods --type worker --json')
            sanitized_output = rdafutils.cleanup_stdout(stdout)
            rdac_pods = json.loads(sanitized_output)
            for pod in rdac_pods:
                if pod.get("hostname") == container_id:
                    logger.info("Worker is up and running")
                    return
            logger.info(f"Waiting for worker to be up and running... retry {retry}")
            time.sleep(10)

        if not rdafutils.query_yes_no(f"Warning: Worker on host {host} may not have been upgraded properly. Continue anyway?"):
            cli_err_exit(f'Worker on host {host} has not been upgraded properly.')
