#!/usr/bin/bash

LOCAL_IP="${MARIADB_NODE_ADDR}"

SYNC_WAIT_TIME=3600
DB_USERNAME="${MARIADB_HEALTHCHK_USERNAME}"
DB_PASSWORD="${MARIADB_HEALTHCHK_PASSWORD}"
DB_MOUNTPOINT="${MARIADB_DATA_DIR}"
MIN_FREESPACE_PERC=20
SSH_USER="${SSH_USER}"
NODE_IP_ADDRESS="${MARIADB_CLUSTER_HOSTS}"

LIST=$(echo $NODE_IP_ADDRESS | tr "," "\n")
A=1
for addr in $LIST
do
    export NODE$A=$(echo $addr)
    if [ "${LOCAL_IP}" = $addr ]
    then
      # the fact that the script is currently running on this node, is a sign that the node is up
      res=0
    else
      /usr/bin/ssh -q -o StrictHostKeyChecking=no "${SSH_USER}"@$addr "uname -a" > /dev/null 2>&1
      res=$?
    fi
    if [ $res -eq 0 ]
    then
        echo "Cluster Node$A $addr: is accessible"
    else
        echo "Cluster Node$A $addr: is not accessible, please check network connectivity or SSH credentials"
        exit
    fi
    A=`expr $A + 1`
done

DB_PASSWORD_OPTION=" -p${DB_PASSWORD}"
if [[ -z "${DB_PASSWORD}" ]]; then
    # no password
    DB_PASSWORD_OPTION=""
fi

/usr/bin/date
echo ""
mariadb_node_space_check () {
  USED_SPACE_PERC=`df -h $DB_MOUNTPOINT | awk ' {if(NR>1) print}' | awk '{print $5}' | cut -f1 -d"%"`
  # shellcheck disable=SC2006
  AVAILABLE_SPACE_PERC=`expr 100 - $USED_SPACE_PERC`
  AVAILABLE_SPACE=`df -h $DB_MOUNTPOINT | awk ' {if(NR>1) print}' | awk '{print $4}' | cut -f1 -d"%"`
  echo "Checking available storage space: $AVAILABLE_SPACE ($AVAILABLE_SPACE_PERC %)"
  if [[ $AVAILABLE_SPACE_PERC -le $MIN_FREESPACE_PERC ]]
  then
    echo "Minimum required free storage space in % is $MIN_FREESPACE_PERC"
    echo "Mariadb Cluster Node is running short on storage space, please increase the storage space and try again"
    exit
  fi
}

mariadb_node_container_check () {
  /usr/bin/docker ps | grep mariadb > /dev/null
  if [ $? -ne 0 ]
  then
	  echo "Mariadb Cluster Node's container id: $MARIADB_NODE_CONTAINER_ID is not in Running state, please check the /var/log/mysql/mysql.log"
	  exit
	fi
}

mariadb_node_status_check () {
  /usr/bin/docker ps | grep mariadb > /dev/null

  if [ $? -ne 0 ]
  then
    echo "Mariadb Cluster Node is DOWN, Restarting it...."
    sudo /usr/bin/sed -i "s/- MARIADB_GALERA_CLUSTER_BOOTSTRAP=yes/#- MARIADB_GALERA_CLUSTER_BOOTSTRAP=yes/g" /opt/rdaf/deployment-scripts/$LOCAL_IP/infra.yaml
    /usr/local/bin/docker-compose --project-name infra -f /opt/rdaf/deployment-scripts/$LOCAL_IP/infra.yaml up -d mariadb
    sleep 30

    /usr/bin/docker ps | grep mariadb > /dev/null
    if [ $? -eq 0 ]
    then
      echo "Mariadb Cluster Node is UP, Checking Replication Status...."
			ITERATIONS=`expr $SYNC_WAIT_TIME / 30`
      A=0
      while [ $A -le $ITERATIONS ]
      do
      mariadb_node_container_check
      sleep 30

      /usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $LOCAL_IP -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
      if [ $? -eq 0 ]
      then
        echo "Mariadb Cluster Node Replication Status: In Sync now"
        sudo /usr/bin/sed -i "s/#- MARIADB_GALERA_CLUSTER_BOOTSTRAP=yes/- MARIADB_GALERA_CLUSTER_BOOTSTRAP=yes/g" /opt/rdaf/deployment-scripts/$LOCAL_IP/infra.yaml
        exit
        else
          echo "Mariadb Cluster Node Replication Status: Sync is in-progress...."
          A=$[$A+1]
      fi
      done

      /usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $LOCAL_IP -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
      if [ $? -eq 1 ]
			then
			  echo "Mariadb Cluster Node Replication Status: Timed Out, Please check the logs in /var/log/mysql/mysql.log"
			fi
    fi
  fi
}

mariadb_cluster_status_check () {
	if [[ "$LOCAL_IP" == "$NODE1" ]]
	then
		/usr/bin/mysql --login-path=local -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $NODE2 -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
		PEER_NODE2_STATUS=`echo $?`
		/usr/bin/mysql --login-path=local -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $NODE3 -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
		PEER_NODE3_STATUS=`echo $?`

		if [[ "PEER_NODE2_STATUS" -eq 0 ]]
		then
			echo "Peer Mariadb Cluster Node $NODE2: UP & Sync"
		else
			echo "Peer Mariadb Cluster Node $NODE2: DOWN"
		fi

		if [[ "PEER_NODE3_STATUS" -eq 0 ]]
    then
      echo "Peer Mariadb Cluster Node $NODE3: UP & Sync"
    else
      echo "Peer Mariadb Cluster Node $NODE3: DOWN"
    fi
		elif [[ "$LOCAL_IP" == "$NODE2" ]]
		then
			/usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $NODE1 -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
      PEER_NODE1_STATUS=`echo $?`
      /usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $NODE3 -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
      PEER_NODE3_STATUS=`echo $?`
			if [[ "PEER_NODE1_STATUS" -eq 0 ]]
      then
        echo "Peer Mariadb Cluster Node $NODE1: UP & Sync"
      else
        echo "Peer Mariadb Cluster Node $NODE1: DOWN"
      fi

      if [[ "PEER_NODE3_STATUS" -eq 0 ]]
      then
        echo "Peer Mariadb Cluster Node $NODE3: UP & Sync"
      else
        echo "Peer Mariadb Cluster Node $NODE3: DOWN"
      fi
		elif [[ "$LOCAL_IP" == "$NODE3" ]]
		then
			/usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $NODE1 -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
      PEER_NODE1_STATUS=`echo $?`
      /usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $NODE2 -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
      PEER_NODE2_STATUS=`echo $?`

			if [[ "PEER_NODE1_STATUS" -eq 0 ]]
      then
        echo "Peer Mariadb Cluster Node $NODE1: UP & Sync"
      else
        echo "Peer Mariadb Cluster Node $NODE1: DOWN"
      fi

      if [[ "PEER_NODE2_STATUS" -eq 0 ]]
      then
        echo "Peer Mariadb Cluster Node $NODE2: UP & Sync"
      else
        echo "Peer Mariadb Cluster Node $NODE2: DOWN"
      fi
		else
			echo "Unknown Node"
			exit
	fi
}

echo "Mariadb Cluster Node: $LOCAL_IP"
docker ps -a | grep mariadb: > /dev/null
if [ $? -eq 0 ]
then
  echo "Mariadb Cluster Node Status: UP"
else
  echo "Mariadb Cluster Node Status: DOWN"
fi

/usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $LOCAL_IP -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
if [ $? -eq 0 ]
then
  echo "Mariadb Cluster Node Replication Status: Sync"
  # hardcoded data mount point for time being
  USED_SPACE_PERC=`df -h $DB_MOUNTPOINT | awk ' {if(NR>1) print}' | awk '{print $5}' | cut -f1 -d"%"`
  AVAILABLE_SPACE_PERC=`expr 100 - $USED_SPACE_PERC`
  AVAILABLE_SPACE=`df -h $DB_MOUNTPOINT | awk ' {if(NR>1) print}' | awk '{print $4}' | cut -f1 -d"%"`
  echo "Checking available storage space: $AVAILABLE_SPACE ($AVAILABLE_SPACE_PERC %)"
  mariadb_cluster_status_check
  exit
else
  echo "Mariadb Cluster Node Replication Status: Out of Sync"
  mariadb_node_space_check
fi

if [[ "$LOCAL_IP" == "$NODE1" ]]
then
  echo "Peer Nodes are $NODE2 and $NODE3"
  /usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $NODE2 -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
  PEER_NODE2_STATUS=`echo $?`
  /usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $NODE3 -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
  PEER_NODE3_STATUS=`echo $?`

  if [[ "$PEER_NODE2_STATUS" -eq 0 ]] && [[ "$PEER_NODE3_STATUS" -eq 0 ]]
  then
    echo "Peer Mariadb Cluster Nodes: $NODE2, $NODE3"
    echo "Peer Mariadb Cluster Nodes Status: UP and Sync"
    mariadb_node_status_check
  else
    echo "Peer Mariadb Cluster Nodes: $NODE2, $NODE3"
    echo "Peer Mariadb Cluster Nodes Status: One or more Peer nodes are down or out of sync, quorum is missing, cannot re-start this node"
  fi

elif [[ "$LOCAL_IP" == "$NODE2" ]]
then
  echo "Peer Mariadb Nodes are $NODE1 and $NODE3"
  /usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $NODE1 -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
  PEER_NODE1_STATUS=`echo $?`

  /usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $NODE3 -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
  PEER_NODE3_STATUS=`echo $?`

  if [[ "$PEER_NODE1_STATUS" -eq 0 ]] && [[ "$PEER_NODE3_STATUS" -eq 0 ]]
  then
    echo "Peer Mariadb Cluster Nodes: $NODE1, $NODE3"
    echo "Peer Mariadb Cluster Nodes Status: UP and Sync"
    mariadb_node_status_check
  else
    echo "Peer Mariadb Cluster Nodes: $NODE1, $NODE3"
    echo "Peer Mariadb Cluster Nodes Status: One or more Peer nodes are down or out of sync, quorum is missing, cannot re-start this node"
  fi
elif [[ "$LOCAL_IP" == "$NODE3" ]]
then
  echo "Peer Mariadb Nodes are $NODE1 and $NODE2"
  /usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $NODE1 -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
  PEER_NODE1_STATUS=`echo $?`
  /usr/bin/mysql -u $DB_USERNAME --password=$DB_PASSWORD_OPTION -h $NODE2 -P 3306 -e "show status like 'wsrep_local_state_comment';" | grep Synced > /dev/null
  PEER_NODE2_STATUS=`echo $?`

  if [[ "$PEER_NODE1_STATUS" -eq 0 ]] && [[ "$PEER_NODE2_STATUS" -eq 0 ]]
  then
    echo "Peer Mariadb Cluster Nodes: $NODE1, $NODE2"
    echo "Peer Mariadb Cluster Nodes Status: UP and Sync"
    mariadb_node_status_check
  else
    echo "Peer Mariadb Cluster Nodes: $NODE1, $NODE2"
    echo "Peer Mariadb Cluster Nodes Status: One or more Peer nodes are down or out of sync, quorum is missing, cannot re-start this node"
  fi
else
  echo "Unknown Node"
  exit
fi