#!/usr/bin/env python3

'''
Copyright (c) 2022 Cloudfabrix Software Inc. All rights reserved.

This program is a wrapper to setup docker based environment to run RDA Client command line tool.

Requirements to run this script:
- python3
- docker
'''
import configparser
import json
import os
import sys
import re
import platform
import logging

logger = logging.getLogger(__name__)


actual_config_path = None
user_data_path = None


def check_dependencies():
    global actual_config_path
    global user_data_path

    # Check OS
    osname = platform.system()
    logger.info(f"Detected OS Name: {osname}")

    # Check for python  3
    if sys.version_info < (3, 0):
        logger.error("ERROR: This script should be run with any python3 environment.")
        print ()
        sys.exit(1)

    # Check for docker
    strm = os.popen("docker --version")
    verstr = strm.read()

    expr = "Docker version (.*),.*"
    m = re.match(expr, verstr)
    if not m:
        logger.error("Error: docker not detected on this system. Docker must be installed to run rdac command")
        sys.exit(1)
    
    logger.info("Detected docker version: {}".format(m.group(1)))

    # Check for RDA Configuration
    env_path = os.environ.get("RDA_NETWORK_CONFIG")
    if not env_path:
        if os.path.exists(os.path.join('/opt/rdaf/config/network_config/external-config.json')):
            env_path = '/opt/rdaf/config/network_config/external-config.json'
        else:
            env_path = '/opt/rdaf/config/network_config/config.json'
    if not os.path.isfile(env_path):
        logger.error(f"Error: file specified by ENV variable: RDA_NETWORK_CONFIG does not exist: {env_path}")
        logger.error("")
        logger.error("RDA Configuration JSON file should be placed under your home directory "
                     "path: ~/.rda/rda_network_config.json")
        logger.error("")
        logger.error("or ")
        logger.error("")
        logger.error("A path to that file must be specified using Env variable RDA_NETWORK_CONFIG")
        logger.error("")
        sys.exit(1)

    actual_config_path = env_path

    try:
        json.loads(open(actual_config_path).read())
    except Exception as e:
        logger.error(e)
        logger.error(f"Error: RDA Configuration file '{actual_config_path}' is not a valid JSON file")
        logger.error("")
        sys.exit(1)
    
    user_data_path = os.path.expanduser('~/rdac_data/')
    if not os.path.isdir(user_data_path):
        logger.info(f"Creating Data directory: {user_data_path}")
        try:
            os.makedirs(user_data_path)
        except:
            logger.error(f"Error: Failed to create data directory: {user_data_path}")
            sys.exit(1)


def read_portal_attributes_from_config_file(config_path='/opt/rdaf/rdaf.cfg'):
    """
    Read configuration file and extract specified attributes.
    Returns:
        the attributes (if found)
    """
    config = configparser.ConfigParser(allow_no_value=True)

    try:
        config.read(config_path)
        portal_token = config.get('common', 'portal_token', fallback=None)
        internal_adv_host = config.get('haproxy', 'advertised_internal_host', fallback=None)
        external_adv_host = config.get('haproxy', 'advertised_external_host', fallback=None)
        haproxy_host = config.get('haproxy', 'host', fallback=None)
        portal_host = internal_adv_host or external_adv_host or haproxy_host
        return portal_token, portal_host
    except configparser.Error as e:
        return None, None

def run():
    check_dependencies()

    args = sys.argv[1:]
    cwd = os.getcwd()
    mount1 = f"{actual_config_path}:/root/.rda/rda_network_config.json"
    mount2 = f"{user_data_path}:/data/"
    mount3 = f"{cwd}:/home/"
    image = "${REGISTRY}/ubuntu-rdac:${TAG}"

    portal_token, portal_host = read_portal_attributes_from_config_file()
    env = ''
    if portal_token:
        env += f' -e PORTAL_TOKEN={portal_token} '
    if portal_host:
        env += f' -e PORTAL_HOST={portal_host} '
    env += f" -e RDA_NETWORK_CONFIG=/root/.rda/rda_network_config.json"
    extra_args = ""
    if len(args) > 0:
        if args[0] == "update":
            logger.info("Updating docker image with tag ${TAG}...")
            status = os.system(f"docker pull {image}")
            if status != 0:
                logger.warning("Failed to download latest docker image")
                sys.exit(1)
            sys.exit(0)

        if args[0] != "shell":
            fargs = []
            for a in args:
                a = a.replace(' ', '\\ ')
                fargs.append(a)
            extra_args = "rdac {}".format(' '.join(fargs))

    command = f"docker run --log-driver json-file --log-opt max-size=10m --log-opt max-file=5 --rm -i {env} -v {mount1} -v {mount2} -v {mount3} {image} {extra_args}"
    os.system(command)


if __name__ == "__main__":
    run()
