#!/usr/bin/env python3

'''
Copyright (c) 2022 Cloudfabrix Software Inc. All rights reserved.

This program is a wrapper to setup containerd (ctr) based environment to run RDA Client command line tool.

Requirements to run this script:
- python3
- containerd CLI (ctr)
'''
import configparser
import json
import os
import sys
import platform
import logging
import time

logger = logging.getLogger(__name__)

actual_config_path = None
user_data_path = None


def _ctr_available() -> bool:
    try:
        stream = os.popen('ctr version')
        verstr = stream.read()
        # Typical output contains both Client: and Server:
        if 'Client:' in verstr and 'Server:' in verstr:
            logger.info("Detected containerd (ctr):")
        return True
    except Exception:
        return False


def check_dependencies():
    global actual_config_path
    global user_data_path

    # Check OS
    osname = platform.system()
    logger.info(f"Detected OS Name: {osname}")

    # Check for python 3
    if sys.version_info < (3, 0):
        logger.error("ERROR: This script should be run with any python3 environment.")
        print()
        sys.exit(1)

    # Check for containerd (ctr)
    if not _ctr_available():
        logger.error("Error: containerd (ctr) not detected on this system. It must be installed to run rdac command")
        sys.exit(1)

    # Check for RDA Configuration
    env_path = os.environ.get("RDA_NETWORK_CONFIG")
    if not env_path:
        if os.path.exists(os.path.join('/opt/rdaf/config/network_config/external-config.json')):
            env_path = '/opt/rdaf/config/network_config/external-config.json'
        else:
            env_path = '/opt/rdaf/config/network_config/config.json'
    if not os.path.isfile(env_path):
        logger.error(f"Error: file specified by ENV variable: RDA_NETWORK_CONFIG does not exist: {env_path}")
        logger.error("")
        logger.error("RDA Configuration JSON file should be placed under your home directory "
                     "path: ~/.rda/rda_network_config.json")
        logger.error("")
        logger.error("or ")
        logger.error("")
        logger.error("A path to that file must be specified using Env variable RDA_NETWORK_CONFIG")
        logger.error("")
        sys.exit(1)

    actual_config_path = env_path

    try:
        json.loads(open(actual_config_path).read())
    except Exception as e:
        logger.error(e)
        logger.error(f"Error: RDA Configuration file '{actual_config_path}' is not a valid JSON file")
        logger.error("")
        sys.exit(1)

    user_data_path = os.path.expanduser('~/rdac_data/')
    if not os.path.isdir(user_data_path):
        logger.info(f"Creating Data directory: {user_data_path}")
        try:
            os.makedirs(user_data_path)
        except Exception:
            logger.error(f"Error: Failed to create data directory: {user_data_path}")
            sys.exit(1)


def read_portal_attributes_from_config_file(config_path='/opt/rdaf/rdaf.cfg'):
    """
    Read configuration file and extract specified attributes.
    Returns:
        the attributes (if found)
    """
    config = configparser.ConfigParser(allow_no_value=True)

    try:
        config.read(config_path)
        portal_token = config.get('common', 'portal_token', fallback=None)
        internal_adv_host = config.get('haproxy', 'advertised_internal_host', fallback=None)
        external_adv_host = config.get('haproxy', 'advertised_external_host', fallback=None)
        haproxy_host = config.get('haproxy', 'host', fallback=None)
        portal_host = internal_adv_host or external_adv_host or haproxy_host
        return portal_token, portal_host
    except configparser.Error:
        return None, None


def run():
    check_dependencies()

    args = sys.argv[1:]
    cwd = os.getcwd()

    # Mounts (host -> container)
    # Note: ctr uses: --mount type=bind,src=<host>,dst=<container>,options=rbind:rw
    mounts = [
        (actual_config_path, "/root/.rda/rda_network_config.json"),
        (user_data_path, "/data/"),
        (cwd, "/home/"),
    ]
    mounts_arg = ""
    for src, dst in mounts:
        mounts_arg += f" --mount type=bind,src={src},dst={dst},options=rbind:rw"

    image = "${REGISTRY}/ubuntu-rdac:${TAG}"

    # Environment for ctr: --env KEY=VALUE
    portal_token, portal_host = read_portal_attributes_from_config_file()
    env_args = ""
    if portal_token:
        env_args += f" --env PORTAL_TOKEN={portal_token}"
    if portal_host:
        env_args += f" --env PORTAL_HOST={portal_host}"
    env_args += " --env RDA_NETWORK_CONFIG=/root/.rda/rda_network_config.json"
    env_args += " --env RDA_NATS_OVER_GOLANG=0"

    # Extra args passed to the container entrypoint
    extra_args = ""
    if len(args) > 0:
        if args[0] == "update":
            logger.info("Updating container image with tag ${TAG} using containerd (ctr)...")
            status = os.system(f"sudo ctr image pull --user ${USER}:${PASSWORD} {image}")
            if status != 0:
                logger.warning("Failed to download latest container image")
                sys.exit(1)
            sys.exit(0)

        if args[0] != "shell":
            escaped = []
            for a in args:
                escaped.append(a.replace(' ', '\\ '))
            extra_args = "rdac {}".format(' '.join(escaped))

    # Pull image with authentication before running
    logger.info(f"Pulling container image {image}...")
    pull_status = os.system(f"sudo ctr image pull --user ${USER}:${PASSWORD} {image}")
    if pull_status != 0:
        logger.error("Failed to pull container image. Check registry credentials and connectivity.")
        sys.exit(1)

    # ctr run requires a container name
    container_name = f"rdac-{int(time.time())}"

    # Final command: ctr run IMAGE CONTAINER_NAME [COMMAND...]
    command = f"sudo ctr run --net-host --rm -t{env_args}{mounts_arg} {image} {container_name} {extra_args}"
    logger.info(f"Running: {command}")
    os.system(command)


if __name__ == "__main__":
    run()
